'use strict';
import * as helpers from './helpers';
/**
 * Promise that can be used as a handle. It will not raise errors when rejected until it is explicitly awaited or
 * catch is set
 * @template T Promise result type
 * @implements {Promise}
 */ export class HandlePromise {
    /**
   * Returns whether the promise has resolved
   * @returns {boolean} whether the promise has resolved
   */ get resolved() {
        return !!this._promise.resolved;
    }
    /**
   * Returns whether the promise has rejected
   * @returns {boolean} whether the promise has rejected
   */ get rejected() {
        return !!this._promise.rejected;
    }
    /**
   * Returns whether the promise has completed (either resolved or rejected)
   * @returns {boolean} whether the promise has completed (either resolved or rejected)
   */ get completed() {
        return !!this._promise.completed;
    }
    /**
   * Constructs instance
   * @param {Promise<T>} [promise] original promise to wrap into the handle
   */ constructor(promise){
        this._promise = helpers.createHandlePromise();
        this.then = this._promise.then.bind(this._promise);
        this.catch = this._promise.catch.bind(this._promise);
        this.finally = this._promise.finally.bind(this._promise);
        this.resolve = this._promise.resolve.bind(this._promise);
        this.reject = this._promise.reject.bind(this._promise);
        this.timeout = this._promise.timeout.bind(this._promise);
        if (promise) {
            promise.then(this.resolve).catch(()=>{});
            promise.catch(this.reject);
        }
    }
}
/**
 * Represents a handle promise, resolving when needed condition is reached
 */ export class ConditionPromise extends HandlePromise {
    /**
   * Calls the predicate to check if the promise should be resolved
   */ check() {
        if (this._predicate()) {
            this.resolve();
        }
    }
    /**
   * Constructs instance
   * @param {() => boolean} predicate function that returns true when the condition is reached
   */ constructor(predicate){
        super();
        this._predicate = predicate;
        this.check();
    }
}

//# sourceMappingURL=data:application/json;base64,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